"""Profile data manager for local storage"""

import json
import os
from pathlib import Path
from dataclasses import dataclass, asdict, field
from typing import Optional, Tuple
import shutil
import re


@dataclass
class LocationData:
    """Location data from map selection"""
    latitude: float = 0.0
    longitude: float = 0.0
    formatted_address: str = ""

    def is_valid(self) -> bool:
        """Check if location is set"""
        return self.latitude != 0.0 or self.longitude != 0.0

    def to_dict(self) -> dict:
        return asdict(self)

    @classmethod
    def from_dict(cls, data: dict) -> 'LocationData':
        if not data:
            return cls()
        return cls(
            latitude=data.get('latitude', 0.0),
            longitude=data.get('longitude', 0.0),
            formatted_address=data.get('formatted_address', '')
        )


@dataclass
class ProfileData:
    """Profile data structure"""
    supermarket_name: str = ""
    location: LocationData = field(default_factory=LocationData)
    phone_number: str = ""
    logo_path: str = ""

    def is_complete(self) -> bool:
        """Check if all required fields are filled"""
        return all([
            self.supermarket_name.strip(),
            self.location.is_valid(),
            self.phone_number.strip()
        ])

    def to_dict(self) -> dict:
        """Convert to dictionary"""
        return {
            'supermarket_name': self.supermarket_name,
            'location': self.location.to_dict(),
            'phone_number': self.phone_number,
            'logo_path': self.logo_path
        }

    @classmethod
    def from_dict(cls, data: dict) -> 'ProfileData':
        """Create from dictionary"""
        location_data = data.get('location', {})
        if isinstance(location_data, dict):
            location = LocationData.from_dict(location_data)
        else:
            location = LocationData()

        return cls(
            supermarket_name=data.get('supermarket_name', ''),
            location=location,
            phone_number=data.get('phone_number', ''),
            logo_path=data.get('logo_path', '')
        )


@dataclass
class AppSettings:
    """Application settings including storage path"""
    storage_path: str = ""

    def to_dict(self) -> dict:
        return asdict(self)

    @classmethod
    def from_dict(cls, data: dict) -> 'AppSettings':
        return cls(storage_path=data.get('storage_path', ''))


class ProfileManager:
    """Manages profile data storage and retrieval"""

    DEFAULT_APP_DATA_DIR = Path.home() / ".cartify"
    SETTINGS_FILE = Path.home() / ".cartify" / "settings.json"

    def __init__(self):
        self._settings: Optional[AppSettings] = None
        self._profile: Optional[ProfileData] = None
        self._ensure_base_directory()
        self._load_settings()

    def _ensure_base_directory(self):
        """Create base directory for settings"""
        self.DEFAULT_APP_DATA_DIR.mkdir(parents=True, exist_ok=True)

    @property
    def storage_path(self) -> Path:
        """Get current storage path"""
        if self._settings and self._settings.storage_path:
            return Path(self._settings.storage_path)
        return self.DEFAULT_APP_DATA_DIR

    @property
    def profile_file(self) -> Path:
        """Get profile file path"""
        return self.storage_path / "profile.json"

    @property
    def logo_dir(self) -> Path:
        """Get logo directory path"""
        return self.storage_path / "logos"

    def _load_settings(self):
        """Load application settings"""
        try:
            if self.SETTINGS_FILE.exists():
                with open(self.SETTINGS_FILE, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                    self._settings = AppSettings.from_dict(data)
            else:
                self._settings = AppSettings()
        except Exception as e:
            print(f"⚠️ Error loading settings: {e}")
            self._settings = AppSettings()

        self._ensure_storage_directories()

    def _ensure_storage_directories(self):
        """Create storage directories if they don't exist"""
        self.storage_path.mkdir(parents=True, exist_ok=True)
        self.logo_dir.mkdir(parents=True, exist_ok=True)

    def save_settings(self, settings: AppSettings) -> bool:
        """Save application settings"""
        try:
            new_path = Path(settings.storage_path) if settings.storage_path else self.DEFAULT_APP_DATA_DIR
            new_path.mkdir(parents=True, exist_ok=True)
            (new_path / "logos").mkdir(parents=True, exist_ok=True)

            with open(self.SETTINGS_FILE, 'w', encoding='utf-8') as f:
                json.dump(settings.to_dict(), f, indent=2)

            self._settings = settings
            self._ensure_storage_directories()

            print(f"✅ Settings saved. Storage path: {self.storage_path}")
            return True
        except Exception as e:
            print(f"❌ Error saving settings: {e}")
            return False

    def get_settings(self) -> AppSettings:
        """Get current settings"""
        return self._settings or AppSettings()

    def set_storage_path(self, path: str) -> bool:
        """Change storage path and migrate data"""
        old_path = self.storage_path
        new_path = Path(path)

        try:
            new_path.mkdir(parents=True, exist_ok=True)
            (new_path / "logos").mkdir(parents=True, exist_ok=True)

            if old_path.exists() and old_path != new_path:
                old_profile = old_path / "profile.json"
                if old_profile.exists():
                    shutil.copy2(old_profile, new_path / "profile.json")

                old_logos = old_path / "logos"
                if old_logos.exists():
                    for logo_file in old_logos.glob("*"):
                        shutil.copy2(logo_file, new_path / "logos" / logo_file.name)

            settings = AppSettings(storage_path=str(new_path))
            self.save_settings(settings)

            self._profile = None

            return True
        except Exception as e:
            print(f"❌ Error changing storage path: {e}")
            return False

    def load_profile(self) -> ProfileData:
        """Load profile from local storage"""
        if self._profile is not None:
            return self._profile

        try:
            if self.profile_file.exists():
                with open(self.profile_file, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                    self._profile = ProfileData.from_dict(data)
            else:
                self._profile = ProfileData()
        except (json.JSONDecodeError, IOError) as e:
            print(f"⚠️ Error loading profile: {e}")
            self._profile = ProfileData()

        return self._profile

    def save_profile(self, profile: ProfileData) -> bool:
        """Save profile to local storage"""
        try:
            self._ensure_storage_directories()

            with open(self.profile_file, 'w', encoding='utf-8') as f:
                json.dump(profile.to_dict(), f, indent=2, ensure_ascii=False)

            self._profile = profile
            print(f"✅ Profile saved to: {self.profile_file}")
            return True
        except IOError as e:
            print(f"❌ Error saving profile: {e}")
            return False

    def save_logo(self, source_path: str) -> Optional[str]:
        """Copy logo to app data directory and return new path"""
        try:
            source = Path(source_path)
            if not source.exists():
                return None

            self._ensure_storage_directories()

            extension = source.suffix.lower()
            if extension not in ['.png', '.jpg', '.jpeg', '.webp', '.svg']:
                extension = '.png'

            dest_filename = f"store_logo{extension}"
            dest_path = self.logo_dir / dest_filename

            for old_logo in self.logo_dir.glob("store_logo.*"):
                old_logo.unlink()

            shutil.copy2(source_path, dest_path)
            print(f"✅ Logo saved to: {dest_path}")

            return str(dest_path)
        except Exception as e:
            print(f"❌ Error saving logo: {e}")
            return None

    def delete_logo(self) -> bool:
        """Delete stored logo"""
        try:
            for logo_file in self.logo_dir.glob("store_logo.*"):
                logo_file.unlink()
            return True
        except Exception as e:
            print(f"❌ Error deleting logo: {e}")
            return False

    def get_storage_path_str(self) -> str:
        """Get the storage directory path as string"""
        return str(self.storage_path)

    def clear_cache(self):
        """Clear cached profile data"""
        self._profile = None

    @staticmethod
    def validate_phone(phone: str) -> Tuple[bool, str]:
        """Validate phone number format"""
        cleaned = re.sub(r'[\s\-\(\)]', '', phone)

        if not cleaned:
            return False, "Phone number is required"

        if not re.match(r'^\+?[\d]{7,15}$', cleaned):
            return False, "Invalid phone format. Use: +1234567890 or 1234567890"

        return True, ""

    @staticmethod
    def validate_name(name: str) -> Tuple[bool, str]:
        """Validate supermarket name"""
        name = name.strip()

        if not name:
            return False, "Supermarket name is required"

        if len(name) < 2:
            return False, "Name must be at least 2 characters"

        if len(name) > 100:
            return False, "Name must be less than 100 characters"

        return True, ""

    @staticmethod
    def validate_location(location: LocationData) -> Tuple[bool, str]:
        """Validate location data"""
        if not location or not location.is_valid():
            return False, "Please select a location on the map"

        if not location.formatted_address.strip():
            return False, "Address not found for this location"

        return True, ""