(function () {
    'use strict';

    const MAX_RETRIES = 20;
    const RETRY_DELAY = 300;
    let retryCount = 0;
    let currentQRTab = 'cert';

    // Get server IP from the page
    function getServerIP() {
        const ipElement = document.getElementById('server-ip');
        if (ipElement) {
            return ipElement.textContent.trim();
        }
        return window.location.hostname;
    }

    // Get QR configurations with correct URLs
    function getQRConfigs() {
        const serverIP = getServerIP();
        return {
            cert: {
                containerId: 'qr-cert',
                inputId: 'cert-url',
                // HTTP for certificate download (no SSL issues!)
                url: `http://${serverIP}:8443/ca.crt`,
                colorDark: '#059669',
                label: 'Certificate Download (HTTP)'
            },
            app: {
                containerId: 'qr-app',
                inputId: 'app-url',
                // HTTPS for main app
                url: `https://${serverIP}`,
                colorDark: '#4f46e5',
                label: 'Main App (HTTPS)'
            }
        };
    }

    // Generate a single QR code
    function generateQRCode(key) {
        const configs = getQRConfigs();
        const config = configs[key];
        if (!config) return false;

        const container = document.getElementById(config.containerId);
        const urlInput = document.getElementById(config.inputId);

        if (!container) {
            console.error(`Container not found for ${config.label}`);
            return false;
        }

        // Update input value with correct URL
        if (urlInput) {
            urlInput.value = config.url;
        }

        const url = config.url;
        container.innerHTML = '';

        try {
            new QRCode(container, {
                text: url,
                width: 220,
                height: 220,
                colorDark: config.colorDark,
                colorLight: "#ffffff",
                correctLevel: QRCode.CorrectLevel.H
            });
            console.log(`QR Code generated for ${config.label}:`, url);
            return true;
        } catch (err) {
            console.error(`Failed to generate QR code for ${config.label}:`, err);
            fallbackToText(container, url);
            return false;
        }
    }

    // Generate all QR codes
    function generateAllQRCodes() {
        if (typeof QRCode === 'undefined') {
            retryCount++;
            if (retryCount < MAX_RETRIES) {
                console.log(`QRCode lib not ready... retry ${retryCount}/${MAX_RETRIES}`);
                const configs = getQRConfigs();
                Object.keys(configs).forEach(key => {
                    const container = document.getElementById(configs[key].containerId);
                    if (container) {
                        container.innerHTML = '<p class="qr-loading">Loading QR...</p>';
                    }
                });
                setTimeout(generateAllQRCodes, RETRY_DELAY);
                return;
            } else {
                console.error('QRCode library failed to load');
                const configs = getQRConfigs();
                Object.keys(configs).forEach(key => {
                    const config = configs[key];
                    const container = document.getElementById(config.containerId);
                    if (container) {
                        fallbackToText(container, config.url);
                    }
                });
                return;
            }
        }

        const configs = getQRConfigs();
        Object.keys(configs).forEach(key => generateQRCode(key));
    }

    // Fallback display when QR library fails
    function fallbackToText(container, url) {
        container.innerHTML = `
            <div class="qr-fallback">
                <p>QR Code unavailable</p>
                <code>${url}</code>
            </div>`;
    }

    // Update all URLs on the page
    function updatePageURLs() {
        const serverIP = getServerIP();
        
        // Update cert URL input
        const certUrlInput = document.getElementById('cert-url');
        if (certUrlInput) {
            certUrlInput.value = `http://${serverIP}:8443/ca.crt`;
        }
        
        // Update app URL input
        const appUrlInput = document.getElementById('app-url');
        if (appUrlInput) {
            appUrlInput.value = `https://${serverIP}`;
        }
        
        // Update download button href
        const downloadBtn = document.querySelector('.cert-btn[download]');
        if (downloadBtn) {
            downloadBtn.href = `/ca.crt`;
        }
        
        // Update app open button href
        const appBtn = document.querySelector('.app-btn');
        if (appBtn) {
            appBtn.href = `https://${serverIP}`;
        }
        
        // Update any links in instructions
        document.querySelectorAll('a[href*="{{LOCAL_IP}}"]').forEach(link => {
            link.href = link.href.replace('{{LOCAL_IP}}', serverIP);
        });
        
        console.log('Page URLs updated for IP:', serverIP);
    }

    // QR Tab switching
    function setupQRTabs() {
        const tabBtns = document.querySelectorAll('.qr-tab-btn');
        const panes = document.querySelectorAll('.qr-pane');
        const dots = document.querySelectorAll('.qr-dot');
        const prevBtn = document.getElementById('qr-prev');
        const nextBtn = document.getElementById('qr-next');

        function switchToTab(tabKey) {
            currentQRTab = tabKey;

            // Update tab buttons
            tabBtns.forEach(btn => {
                btn.classList.toggle('active', btn.getAttribute('data-qr') === tabKey);
            });

            // Update panes
            panes.forEach(pane => {
                pane.classList.toggle('active', pane.id === `qr-pane-${tabKey}`);
            });

            // Update dots
            dots.forEach(dot => {
                dot.classList.toggle('active', dot.getAttribute('data-qr') === tabKey);
            });

            // Update nav buttons
            if (prevBtn && nextBtn) {
                prevBtn.disabled = (tabKey === 'cert');
                nextBtn.disabled = (tabKey === 'app');
                
                // Update button text
                if (tabKey === 'cert') {
                    prevBtn.textContent = '← Certificate';
                    nextBtn.textContent = 'Open App →';
                } else {
                    prevBtn.textContent = '← Download Cert';
                    nextBtn.textContent = 'App →';
                }
            }
        }

        // Tab button clicks
        tabBtns.forEach(btn => {
            btn.addEventListener('click', () => {
                switchToTab(btn.getAttribute('data-qr'));
            });
        });

        // Dot clicks
        dots.forEach(dot => {
            dot.addEventListener('click', () => {
                switchToTab(dot.getAttribute('data-qr'));
            });
        });

        // Navigation buttons
        if (prevBtn) {
            prevBtn.addEventListener('click', () => {
                if (currentQRTab === 'app') switchToTab('cert');
            });
        }

        if (nextBtn) {
            nextBtn.addEventListener('click', () => {
                if (currentQRTab === 'cert') switchToTab('app');
            });
        }

        // Swipe support for mobile
        const qrContent = document.querySelector('.qr-tab-content');
        if (qrContent) {
            let startX = 0;

            qrContent.addEventListener('touchstart', (e) => {
                startX = e.touches[0].clientX;
            }, { passive: true });

            qrContent.addEventListener('touchend', (e) => {
                const endX = e.changedTouches[0].clientX;
                const diff = startX - endX;

                if (Math.abs(diff) > 50) {
                    if (diff > 0 && currentQRTab === 'cert') {
                        switchToTab('app');
                    } else if (diff < 0 && currentQRTab === 'app') {
                        switchToTab('cert');
                    }
                }
            }, { passive: true });
        }
    }

    // Device instruction tabs
    function setupDeviceTabs() {
        const tabs = document.querySelectorAll('.tab-btn');
        const panes = document.querySelectorAll('.tab-pane');

        tabs.forEach(tab => {
            tab.addEventListener('click', () => {
                const device = tab.getAttribute('data-device');

                tabs.forEach(t => t.classList.remove('active'));
                panes.forEach(p => p.classList.remove('active'));

                tab.classList.add('active');
                const pane = document.getElementById(device + '-instructions');
                if (pane) pane.classList.add('active');
            });
        });
    }

    // Copy buttons
    function setupCopyButtons() {
        const copyBtns = document.querySelectorAll('.copy-btn');

        copyBtns.forEach(btn => {
            btn.addEventListener('click', async () => {
                const targetId = btn.getAttribute('data-target');
                const input = document.getElementById(targetId);
                if (!input) return;

                try {
                    await navigator.clipboard.writeText(input.value);
                    showCopyFeedback(btn, true);
                } catch (e) {
                    // Fallback for older browsers
                    input.select();
                    input.setSelectionRange(0, 99999);
                    document.execCommand('copy');
                    showCopyFeedback(btn, true);
                }
            });
        });
    }

    function showCopyFeedback(btn, success) {
        const originalHTML = btn.innerHTML;
        btn.innerHTML = success ? '✓' : '✗';
        btn.classList.add('copied');
        btn.disabled = true;

        setTimeout(() => {
            btn.innerHTML = originalHTML;
            btn.classList.remove('copied');
            btn.disabled = false;
        }, 1500);
    }

    // Auto-detect device and switch to appropriate tab
    function detectDevice() {
        const ua = navigator.userAgent.toLowerCase();
        let device = 'windows';

        if (/android/.test(ua)) device = 'android';
        else if (/iphone|ipad|ipod/.test(ua)) device = 'ios';
        else if (/macintosh|mac os x/.test(ua)) device = 'macos';

        const tab = document.querySelector(`.tab-btn[data-device="${device}"]`);
        if (tab) tab.click();

        console.log('Detected device:', device);
    }

    // Show current protocol info
    function showProtocolInfo() {
        const isHTTP = window.location.protocol === 'http:';
        const protocolBadge = document.querySelector('.current-protocol');
        
        if (protocolBadge) {
            protocolBadge.textContent = isHTTP ? 'HTTP (Safe for download)' : 'HTTPS';
            protocolBadge.className = `current-protocol ${isHTTP ? 'http' : 'https'}`;
        }
        
        console.log('Current protocol:', window.location.protocol);
    }

    // Initialize everything
    function init() {
        console.log('Initializing Cartify Certificate Page...');
        
        updatePageURLs();
        setupQRTabs();
        setupDeviceTabs();
        setupCopyButtons();
        detectDevice();
        showProtocolInfo();
        generateAllQRCodes();
        
        console.log('Initialization complete');
    }

    // Start when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }

    // Expose for debugging
    window.regenerateQRCodes = generateAllQRCodes;
    window.getServerIP = getServerIP;
})();