#Requires -Version 5.1
<#
.SYNOPSIS
    Cartify Local Network HTTPS Setup Script
.PARAMETER Force
    Force regeneration of certificates.
.PARAMETER SkipCertInstall
    Skip installing CA certificate to Windows trust store.
.PARAMETER GpuProfile
    Enable GPU profile for Docker Compose.
.PARAMETER ComposeFile
    Path to Docker Compose file. Default: docker-compose.yaml
.PARAMETER SkipBrowserOpen
    Skip opening the browser automatically.
#>

[CmdletBinding()]
param(
    [switch]$Force,
    [switch]$SkipCertInstall,
    [switch]$GpuProfile,
    [switch]$SkipBrowserOpen,
    [string]$ComposeFile = "docker-compose.yaml",
    [int]$ServiceTimeout = 300,
    [int]$CaddyRetries = 30,
    [int]$ReactRetries = 60,
    [int]$CertDays = 365,
    [int]$CertKeySize = 4096
)

$ErrorActionPreference = "Continue"

#region Configuration

$Script:Version = "2.5.0"
$Script:ProjectRoot = $PSScriptRoot
$Script:TemplatesDir = Join-Path $ProjectRoot "templates"
$Script:RuntimeDir = Join-Path $ProjectRoot ".cartify-runtime"
$Script:CertsDir = Join-Path $RuntimeDir "certs"
$Script:CertPageDir = Join-Path $RuntimeDir "cert-page"
$Script:CaddyDir = Join-Path $RuntimeDir "caddy"
$Script:LogFile = Join-Path $RuntimeDir "setup.log"
$Script:StateFile = Join-Path $RuntimeDir "state.json"
$Script:LocalIP = $null
$Script:Hostname = $null
$Script:PreviousIP = $null
$Script:NeedsCertRegeneration = $false

$Script:CertConfig = @{
    Days    = $CertDays
    KeySize = $CertKeySize
    Country = "US"
    State   = "California"
    City    = "SanFrancisco"
    Org     = "CartifyLocalDev"
    OrgUnit = "Development"
    CAName  = "CartifyLocalCA"
}

$Script:CriticalServices = @("redis", "postgres", "mongodb", "rabbitmq", "gateway", "react-app", "caddy")
$Script:InfraServices = @("redis", "postgres", "mongodb", "rabbitmq", "qdrant")
$Script:BackendServices = @("authentication_gateway", "retailer_user_api", "retailer_product_api", "retailer_settings_api", "retailer_checkout_api", "retailer_notification_api", "retailer_discount_api", "retailer_inventory_api", "gateway")
$Script:CertificateFiles = @("ca.key", "ca.crt", "ca.srl", "server.key", "server.crt", "server.csr", "openssl.cnf", "server.ext", "generate-certs.sh")

#endregion

#region Helper Functions

function Run-Docker {
    param([string]$Arguments)
    $result = & cmd /c "docker $Arguments 2>&1"
    return $result
}

function Run-DockerCompose {
    param([string]$Arguments)
    $composePath = Join-Path $Script:ProjectRoot $ComposeFile
    $result = & cmd /c "docker compose -f `"$composePath`" $Arguments 2>&1"
    return $result
}

function Get-ContainerStatus {
    param([string]$Name)
    $result = Run-Docker "inspect --format={{.State.Status}} $Name"
    if ($result -match "running") { return "running" }
    if ($result -match "exited") { return "exited" }
    if ($result -match "No such object") { return "not_found" }
    return "unknown"
}

function Test-ContainerExists {
    param([string]$Name)
    $result = Run-Docker "ps -aq -f name=^${Name}$"
    return (-not [string]::IsNullOrWhiteSpace($result))
}

#endregion

#region Logging

function Write-Log {
    param(
        [string]$Message,
        [ValidateSet("INFO", "ERROR", "WARNING", "SUCCESS", "DEBUG")]
        [string]$Level = "INFO"
    )
    
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $logMessage = "[$timestamp] [$Level] $Message"
    
    switch ($Level) {
        "ERROR"   { Write-Host $logMessage -ForegroundColor Red }
        "WARNING" { Write-Host $logMessage -ForegroundColor Yellow }
        "SUCCESS" { Write-Host $logMessage -ForegroundColor Green }
        "DEBUG"   { Write-Host $logMessage -ForegroundColor Gray }
        default   { Write-Host $logMessage -ForegroundColor White }
    }
    
    if (Test-Path (Split-Path $Script:LogFile -Parent)) {
        Add-Content -Path $Script:LogFile -Value $logMessage -ErrorAction SilentlyContinue
    }
}

function Write-Banner {
    param([string]$Title, [string]$Subtitle = "", [ConsoleColor]$Color = "Magenta")
    
    $border = "=" * 60
    Write-Host ""
    Write-Host $border -ForegroundColor $Color
    Write-Host ("  " + $Title) -ForegroundColor $Color
    if ($Subtitle) { Write-Host ("  " + $Subtitle) -ForegroundColor $Color }
    Write-Host $border -ForegroundColor $Color
    Write-Host ""
}

#endregion

#region State Management

function Get-SavedState {
    if (Test-Path $Script:StateFile) {
        try {
            return Get-Content $Script:StateFile -Raw | ConvertFrom-Json
        } catch {
            return $null
        }
    }
    return $null
}

function Save-State {
    @{
        IP          = $Script:LocalIP
        Hostname    = $Script:Hostname
        LastUpdated = (Get-Date).ToString("o")
        Version     = $Script:Version
    } | ConvertTo-Json | Set-Content -Path $Script:StateFile -Encoding UTF8 -ErrorAction SilentlyContinue
}

function Test-CertificateRegeneration {
    Write-Log "Checking if certificate regeneration is needed..."
    
    if ($Force) {
        Write-Log "Force flag specified, will regenerate" "INFO"
        return $true
    }
    
    $caCrt = Join-Path $Script:CertsDir "ca.crt"
    $serverCrt = Join-Path $Script:CertsDir "server.crt"
    $serverKey = Join-Path $Script:CertsDir "server.key"
    
    if (-not (Test-Path $caCrt) -or -not (Test-Path $serverCrt) -or -not (Test-Path $serverKey)) {
        Write-Log "Certificates not found, will generate" "INFO"
        return $true
    }
    
    $savedState = Get-SavedState
    if (-not $savedState) {
        Write-Log "No saved state, will regenerate" "INFO"
        return $true
    }
    
    $Script:PreviousIP = $savedState.IP
    if ($Script:PreviousIP -ne $Script:LocalIP) {
        Write-Log "IP changed from $($Script:PreviousIP) to $($Script:LocalIP), will regenerate" "INFO"
        return $true
    }
    
    try {
        $cert = New-Object System.Security.Cryptography.X509Certificates.X509Certificate2($serverCrt)
        $daysLeft = ($cert.NotAfter - (Get-Date)).Days
        if ($daysLeft -lt 30) {
            Write-Log "Certificate expires in $daysLeft days, will regenerate" "WARNING"
            return $true
        }
        Write-Log "Certificate valid for $daysLeft days" "SUCCESS"
    } catch {
        return $true
    }
    
    Write-Log "No regeneration needed" "SUCCESS"
    return $false
}

#endregion

#region Privilege

function Test-Administrator {
    $identity = [Security.Principal.WindowsIdentity]::GetCurrent()
    $principal = New-Object Security.Principal.WindowsPrincipal($identity)
    return $principal.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
}

function Request-Elevation {
    if (-not (Test-Administrator)) {
        Write-Host "Requesting administrator privileges..." -ForegroundColor Yellow
        
        $args = @("-NoProfile", "-NoExit", "-ExecutionPolicy", "Bypass", "-File", "`"$PSCommandPath`"")
        if ($Force) { $args += "-Force" }
        if ($SkipCertInstall) { $args += "-SkipCertInstall" }
        if ($GpuProfile) { $args += "-GpuProfile" }
        if ($SkipBrowserOpen) { $args += "-SkipBrowserOpen" }
        if ($ComposeFile -ne "docker-compose.yaml") { $args += "-ComposeFile"; $args += "`"$ComposeFile`"" }
        
        try {
            Start-Process powershell.exe -ArgumentList $args -Verb RunAs
            exit 0
        } catch {
            Write-Host "Failed to elevate: $_" -ForegroundColor Red
            exit 1
        }
    }
}

#endregion

#region Prerequisites

function Test-Prerequisites {
    Write-Log "Checking prerequisites..."
    
    $dockerVersion = Run-Docker "version --format {{.Server.Version}}"
    if ($dockerVersion -match "error" -or $dockerVersion -match "Cannot connect") {
        Write-Log "Docker is not running. Please start Docker Desktop." "ERROR"
        return $false
    }
    Write-Log "Docker: $dockerVersion" "SUCCESS"
    
    $composeVersion = Run-Docker "compose version --short"
    Write-Log "Docker Compose: $composeVersion" "SUCCESS"
    
    if (-not (Test-Path $Script:TemplatesDir)) {
        Write-Log "Templates directory not found: $Script:TemplatesDir" "ERROR"
        return $false
    }
    
    $templates = @("Caddyfile.template", "cert-page.html", "cert-page.css", "cert-page.js")
    foreach ($t in $templates) {
        if (-not (Test-Path (Join-Path $Script:TemplatesDir $t))) {
            Write-Log "Missing template: $t" "ERROR"
            return $false
        }
    }
    Write-Log "Templates found" "SUCCESS"
    
    $composePath = Join-Path $Script:ProjectRoot $ComposeFile
    if (-not (Test-Path $composePath)) {
        Write-Log "Compose file not found: $composePath" "ERROR"
        return $false
    }
    Write-Log "Compose file found" "SUCCESS"
    
    return $true
}

function Get-LocalIPAddress {
    Write-Log "Detecting local IP..."
    
    $adapters = Get-NetIPAddress -AddressFamily IPv4 | Where-Object {
        $_.InterfaceAlias -notmatch "Loopback" -and
        $_.IPAddress -notmatch "^127\." -and
        $_.IPAddress -notmatch "^169\.254\."
    } | Sort-Object InterfaceMetric
    
    if ($adapters.Count -eq 0) {
        Write-Log "No network interfaces found" "ERROR"
        return $false
    }
    
    $preferred = $adapters | Where-Object { $_.InterfaceAlias -match "Wi-Fi|Ethernet|Wireless" } | Select-Object -First 1
    if (-not $preferred) { $preferred = $adapters | Select-Object -First 1 }
    
    $Script:LocalIP = $preferred.IPAddress
    $Script:Hostname = $env:COMPUTERNAME.ToLower()
    
    Write-Log "IP: $Script:LocalIP ($($preferred.InterfaceAlias))" "SUCCESS"
    Write-Log "Hostname: $Script:Hostname" "SUCCESS"
    
    return $true
}

#endregion

#region Directories

function Initialize-Directories {
    Write-Log "Initializing directories..."
    
    @($Script:RuntimeDir, $Script:CertsDir, $Script:CertPageDir, $Script:CaddyDir) | ForEach-Object {
        if (-not (Test-Path $_)) {
            New-Item -ItemType Directory -Path $_ -Force | Out-Null
        }
    }
    
    Write-Log "Directories ready" "SUCCESS"
    return $true
}

#endregion

#region Certificates

function Remove-PreviousCertificates {
    Write-Log "Removing previous certificates..."
    
    if (-not (Test-Path $Script:CertsDir)) { return }
    
    foreach ($file in $Script:CertificateFiles) {
        $path = Join-Path $Script:CertsDir $file
        if (Test-Path $path) {
            Remove-Item $path -Force -ErrorAction SilentlyContinue
        }
    }
    
    Get-ChildItem $Script:CertsDir -Filter "*.pem" -ErrorAction SilentlyContinue | Remove-Item -Force -ErrorAction SilentlyContinue
}

function Remove-CACertificateFromStore {
    Write-Log "Removing old CA from trust store..."
    
    try {
        $store = New-Object System.Security.Cryptography.X509Certificates.X509Store("Root", "LocalMachine")
        $store.Open("ReadWrite")
        
        $toRemove = $store.Certificates | Where-Object { $_.Subject -match $Script:CertConfig.CAName }
        foreach ($cert in $toRemove) {
            $store.Remove($cert)
            Write-Log "Removed old CA certificate" "SUCCESS"
        }
        
        $store.Close()
    } catch {
        Write-Log "Could not access certificate store: $_" "WARNING"
    }
}

function New-SelfSignedCertificates {
    Write-Log "Generating SSL certificates..."
    
    Remove-PreviousCertificates
    Remove-CACertificateFromStore
    
    # Create OpenSSL config
    $opensslCnf = @"
[req]
default_bits = $($Script:CertConfig.KeySize)
prompt = no
default_md = sha256
distinguished_name = dn
x509_extensions = v3_ca

[dn]
C = $($Script:CertConfig.Country)
ST = $($Script:CertConfig.State)
L = $($Script:CertConfig.City)
O = $($Script:CertConfig.Org)
OU = $($Script:CertConfig.OrgUnit)
CN = $($Script:CertConfig.CAName)

[v3_ca]
subjectKeyIdentifier = hash
authorityKeyIdentifier = keyid:always,issuer
basicConstraints = critical, CA:true
keyUsage = critical, digitalSignature, cRLSign, keyCertSign
"@
    Set-Content -Path (Join-Path $Script:CertsDir "openssl.cnf") -Value $opensslCnf -Encoding ASCII
    
    # Create server extension
    $serverExt = @"
authorityKeyIdentifier=keyid,issuer
basicConstraints=CA:FALSE
keyUsage = digitalSignature, nonRepudiation, keyEncipherment, dataEncipherment
extendedKeyUsage = serverAuth
subjectAltName = @alt_names

[alt_names]
DNS.1 = localhost
DNS.2 = $($Script:Hostname)
IP.1 = 127.0.0.1
IP.2 = $($Script:LocalIP)
"@
    Set-Content -Path (Join-Path $Script:CertsDir "server.ext") -Value $serverExt -Encoding ASCII
    
    # Create generation script
    $script = @"
#!/bin/sh
set -e
cd /certs
apk add --no-cache openssl > /dev/null 2>&1
openssl genrsa -out ca.key $($Script:CertConfig.KeySize) 2>/dev/null
openssl req -x509 -new -nodes -key ca.key -sha256 -days $($Script:CertConfig.Days) -out ca.crt -config openssl.cnf 2>/dev/null
openssl genrsa -out server.key $($Script:CertConfig.KeySize) 2>/dev/null
openssl req -new -key server.key -out server.csr -subj "/CN=$($Script:LocalIP)" 2>/dev/null
openssl x509 -req -in server.csr -CA ca.crt -CAkey ca.key -CAcreateserial -out server.crt -days $($Script:CertConfig.Days) -sha256 -extfile server.ext 2>/dev/null
chmod 644 *.crt *.csr 2>/dev/null || true
chmod 600 *.key 2>/dev/null || true
echo "Certificates generated for $($Script:LocalIP)"
"@
    $script = $script -replace "`r`n", "`n"
    $scriptPath = Join-Path $Script:CertsDir "generate.sh"
    [System.IO.File]::WriteAllText($scriptPath, $script, [System.Text.Encoding]::ASCII)
    
    # Run in Docker
    $null = Run-Docker "pull alpine:latest"
    
    $drive = $Script:CertsDir.Substring(0, 1).ToLower()
    $pathPart = $Script:CertsDir.Substring(2) -replace "\\", "/"
    $dockerPath = "//${drive}${pathPart}"
    
    $output = Run-Docker "run --rm -v `"${dockerPath}:/certs`" alpine:latest /bin/sh /certs/generate.sh"
    Write-Host $output -ForegroundColor Gray
    
    # Cleanup script
    Remove-Item $scriptPath -Force -ErrorAction SilentlyContinue
    
    # Verify
    $required = @("ca.key", "ca.crt", "server.key", "server.crt")
    foreach ($file in $required) {
        $path = Join-Path $Script:CertsDir $file
        if (-not (Test-Path $path) -or (Get-Item $path).Length -eq 0) {
            Write-Log "Certificate file missing or empty: $file" "ERROR"
            return $false
        }
    }
    
    Write-Log "Certificates generated successfully" "SUCCESS"
    return $true
}

function Install-CACertificate {
    if ($SkipCertInstall) {
        Write-Log "Skipping CA installation (-SkipCertInstall)" "WARNING"
        return $true
    }
    
    Write-Log "Installing CA certificate..."
    
    $caCrt = Join-Path $Script:CertsDir "ca.crt"
    
    try {
        $cert = New-Object System.Security.Cryptography.X509Certificates.X509Certificate2($caCrt)
        $store = New-Object System.Security.Cryptography.X509Certificates.X509Store("Root", "LocalMachine")
        $store.Open("ReadWrite")
        
        $existing = $store.Certificates.Find("FindByThumbprint", $cert.Thumbprint, $false)
        if ($existing.Count -eq 0) {
            $store.Add($cert)
            Write-Log "CA certificate installed" "SUCCESS"
        } else {
            Write-Log "CA certificate already installed" "SUCCESS"
        }
        
        $store.Close()
        return $true
    } catch {
        Write-Log "Failed to install CA: $_" "WARNING"
        return $false
    }
}

#endregion

#region Templates

function Copy-TemplateFiles {
    Write-Log "Rendering templates..."
    
    $vars = @{
        "{{LOCAL_IP}}"  = $Script:LocalIP
        "{{HOSTNAME}}"  = $Script:Hostname
        "{{TIMESTAMP}}" = (Get-Date -Format "yyyy-MM-dd HH:mm:ss")
        "{{VERSION}}"   = $Script:Version
    }
    
    # Caddyfile
    $caddyTemplate = Join-Path $Script:TemplatesDir "Caddyfile.template"
    $caddyDest = Join-Path $Script:CaddyDir "Caddyfile"
    $content = Get-Content $caddyTemplate -Raw
    foreach ($key in $vars.Keys) { $content = $content -replace [regex]::Escape($key), $vars[$key] }
    Set-Content -Path $caddyDest -Value $content -NoNewline -Encoding UTF8
    
    # Static files
    $files = @(
        @{ Src = "cert-page.html"; Dst = "index.html" }
        @{ Src = "cert-page.css"; Dst = "cert-page.css" }
        @{ Src = "cert-page.js"; Dst = "cert-page.js" }
    )
    
    foreach ($f in $files) {
        $src = Join-Path $Script:TemplatesDir $f.Src
        $dst = Join-Path $Script:CertPageDir $f.Dst
        $content = Get-Content $src -Raw
        foreach ($key in $vars.Keys) { $content = $content -replace [regex]::Escape($key), $vars[$key] }
        Set-Content -Path $dst -Value $content -NoNewline -Encoding UTF8
    }
    
    # Copy CA cert
    $caSrc = Join-Path $Script:CertsDir "ca.crt"
    $caDst = Join-Path $Script:CertPageDir "ca.crt"
    if (Test-Path $caSrc) { Copy-Item $caSrc $caDst -Force }
    
    Write-Log "Templates rendered" "SUCCESS"
    return $true
}

#endregion

#region Docker Services

function Stop-ConflictingContainers {
    Write-Log "Stopping conflicting containers..."
    
    foreach ($name in @("caddy", "nginx")) {
        if (Test-ContainerExists $name) {
            $null = Run-Docker "stop $name"
            $null = Run-Docker "rm $name"
            Write-Log "Removed $name" "DEBUG"
        }
    }
}

function Start-DockerServices {
    Write-Log "Starting Docker services..."
    
    $null = Run-DockerCompose "down --remove-orphans"
    
    Write-Log "Pulling images..." "INFO"
    if ($GpuProfile) {
        $null = Run-DockerCompose "--profile gpu pull"
        $null = Run-DockerCompose "--profile gpu up -d --remove-orphans"
    } else {
        $null = Run-DockerCompose "pull"
        $null = Run-DockerCompose "up -d --remove-orphans"
    }
    
    Write-Log "Services started" "SUCCESS"
    return $true
}

function Wait-ForServices {
    Write-Log "Waiting for services..."
    
    # Infrastructure
    $timeout = $ServiceTimeout
    $start = Get-Date
    
    while (((Get-Date) - $start).TotalSeconds -lt $timeout) {
        $running = 0
        foreach ($svc in $Script:InfraServices) {
            if ((Get-ContainerStatus $svc) -eq "running") { $running++ }
        }
        
        Write-Host "`r  Infrastructure: $running/$($Script:InfraServices.Count) running    " -NoNewline
        
        if ($running -eq $Script:InfraServices.Count) {
            Write-Host ""
            Write-Log "Infrastructure ready" "SUCCESS"
            break
        }
        Start-Sleep -Seconds 2
    }
    
    # Wait for migrations
    Write-Log "Waiting for DB migrations..."
    Start-Sleep -Seconds 10
    
    # Backend
    $start = Get-Date
    while (((Get-Date) - $start).TotalSeconds -lt $timeout) {
        $running = 0
        foreach ($svc in $Script:BackendServices) {
            if ((Get-ContainerStatus $svc) -eq "running") { $running++ }
        }
        
        Write-Host "`r  Backend: $running/$($Script:BackendServices.Count) running    " -NoNewline
        
        if ($running -eq $Script:BackendServices.Count) {
            Write-Host ""
            Write-Log "Backend ready" "SUCCESS"
            break
        }
        Start-Sleep -Seconds 3
    }
    Write-Host ""
    
    # React
    Write-Log "Waiting for React app..."
    for ($i = 0; $i -lt $ReactRetries; $i++) {
        if ((Get-ContainerStatus "react-app") -eq "running") {
            $check = Run-Docker "exec react-app wget -q -O - --timeout=3 http://localhost:3000"
            if ($check -match "html|DOCTYPE|root") {
                Write-Log "React app ready" "SUCCESS"
                break
            }
        }
        Write-Host "`r  React: attempt $($i+1)/$ReactRetries    " -NoNewline
        Start-Sleep -Seconds 2
    }
    Write-Host ""
    
    # Caddy - Test HTTP on port 8443 (cert page) and HTTPS on port 443 (main app)
    Write-Log "Waiting for Caddy..."
    for ($i = 0; $i -lt $CaddyRetries; $i++) {
        if ((Get-ContainerStatus "caddy") -eq "running") {
            # Test HTTP port 8443 (cert page - NO SSL)
            $checkCertPage = Run-Docker "exec caddy wget -q -O - http://localhost:8443/health"
            if ($checkCertPage -match "OK") {
                Write-Log "Caddy cert page (HTTP:8443) ready" "SUCCESS"
                
                # Also test HTTPS port 443 (main app)
                $checkMainApp = Run-Docker "exec caddy wget -q -O - --no-check-certificate https://localhost:443/health"
                if ($checkMainApp -match "OK") {
                    Write-Log "Caddy main app (HTTPS:443) ready" "SUCCESS"
                }
                break
            }
        }
        Write-Host "`r  Caddy: attempt $($i+1)/$CaddyRetries    " -NoNewline
        Start-Sleep -Seconds 2
    }
    Write-Host ""
    
    # Final check
    Write-Log "Verifying services..."
    foreach ($svc in $Script:CriticalServices) {
        $status = Get-ContainerStatus $svc
        if ($status -eq "running") {
            Write-Log "  $svc : OK" "SUCCESS"
        } else {
            Write-Log "  $svc : $status" "WARNING"
        }
    }
}

function Test-Connectivity {
    Write-Log "Testing connectivity..."
    
    # Test HTTP port 8443 (cert page - NO SSL)
    $check1 = Run-Docker "exec caddy wget -q -O - http://localhost:8443/health"
    if ($check1 -match "OK") {
        Write-Log "Certificate page (HTTP:8443): OK" "SUCCESS"
    } else {
        Write-Log "Certificate page (HTTP:8443): Failed" "WARNING"
    }
    
    # Test certificate download (HTTP)
    $checkCert = Run-Docker "exec caddy wget -q -O - http://localhost:8443/ca.crt"
    if ($checkCert -match "BEGIN CERTIFICATE") {
        Write-Log "Certificate download (HTTP): OK" "SUCCESS"
    } else {
        Write-Log "Certificate download (HTTP): Failed" "WARNING"
    }
    
    # Test HTTPS port 443 (main app)
    $check2 = Run-Docker "exec caddy wget -q -O - --no-check-certificate https://localhost:443/health"
    if ($check2 -match "OK") {
        Write-Log "Main app health (HTTPS:443): OK" "SUCCESS"
    } else {
        Write-Log "Main app health (HTTPS:443): Failed" "WARNING"
    }
    
    # Test React through HTTPS
    $check3 = Run-Docker "exec caddy wget -q -O - --no-check-certificate https://localhost:443/"
    if ($check3 -match "html|DOCTYPE") {
        Write-Log "React app (HTTPS): OK" "SUCCESS"
    } else {
        Write-Log "React app (HTTPS): May still be loading" "WARNING"
    }
}

#endregion

#region Display

function Show-Status {
    Write-Host ""
    Write-Host "========== SERVICE STATUS ==========" -ForegroundColor Cyan
    $null = Run-DockerCompose "ps"
    Write-Host ""
}

function Show-AccessInfo {
    Write-Banner "SETUP COMPLETE" "Cartify Local HTTPS v$Script:Version" Green
    
    Write-Host "ACCESS URLS:" -ForegroundColor Yellow
    Write-Host ""
    Write-Host "  Certificate Page (HTTP - for download):" -ForegroundColor White
    Write-Host "    http://$Script:LocalIP`:8443" -ForegroundColor Cyan
    Write-Host "    http://$Script:LocalIP`:8443/ca.crt" -ForegroundColor Gray
    Write-Host ""
    Write-Host "  Main Application (HTTPS - after cert install):" -ForegroundColor White
    Write-Host "    https://$Script:LocalIP" -ForegroundColor Cyan
    Write-Host ""
    Write-Host "  Monitoring (HTTPS):" -ForegroundColor White
    Write-Host "    https://$Script:LocalIP/grafana" -ForegroundColor Gray
    Write-Host "    https://$Script:LocalIP/prometheus" -ForegroundColor Gray
    Write-Host ""
    
    if ($Script:PreviousIP -and $Script:PreviousIP -ne $Script:LocalIP) {
        Write-Host "  IP changed: $Script:PreviousIP -> $Script:LocalIP" -ForegroundColor Yellow
        Write-Host ""
    }
    
    Write-Host "MOBILE SETUP:" -ForegroundColor Yellow
    Write-Host "  1. Connect to same Wi-Fi network"
    Write-Host "  2. Open " -NoNewline
    Write-Host "http://$Script:LocalIP`:8443" -ForegroundColor Cyan -NoNewline
    Write-Host " (HTTP - no warning!)"
    Write-Host "  3. Download and install the CA certificate"
    Write-Host "  4. Access " -NoNewline
    Write-Host "https://$Script:LocalIP" -ForegroundColor Cyan -NoNewline
    Write-Host " (HTTPS - secure)"
    Write-Host ""
    
    Write-Host "PROTOCOL INFO:" -ForegroundColor Yellow
    Write-Host "  Port 8443 = HTTP  (certificate download - no SSL warning)" -ForegroundColor White
    Write-Host "  Port 443  = HTTPS (main app - requires certificate)" -ForegroundColor White
    Write-Host "  Port 80   = HTTP  (redirects to HTTPS)" -ForegroundColor White
    Write-Host ""
    
    if (-not $Script:NeedsCertRegeneration) {
        Write-Host "  [Certificates reused - IP unchanged]" -ForegroundColor Green
    } else {
        Write-Host "  [New certificates generated]" -ForegroundColor Yellow
    }
    Write-Host ""
}

function Open-Browser {
    if ($SkipBrowserOpen) { return }
    
    Start-Sleep -Seconds 3
    
    # Open HTTP URL for cert page (not HTTPS!)
    $url = "http://$Script:LocalIP`:8443"
    
    try {
        Start-Process $url
        Write-Log "Browser opened: $url (HTTP)" "SUCCESS"
    } catch {
        Write-Log "Open browser manually: $url" "WARNING"
    }
}

#endregion

#region Main

function Main {
    Clear-Host
    Write-Banner "CARTIFY HTTPS SETUP" "Version $Script:Version"
    
    Request-Elevation
    
    Write-Log "Starting setup..."
    
    # Prerequisites
    if (-not (Test-Prerequisites)) {
        Read-Host "Press Enter to exit"
        return
    }
    
    # Directories
    if (-not (Initialize-Directories)) {
        Read-Host "Press Enter to exit"
        return
    }
    
    # IP Detection
    if (-not (Get-LocalIPAddress)) {
        Read-Host "Press Enter to exit"
        return
    }
    
    # Certificate check
    $Script:NeedsCertRegeneration = Test-CertificateRegeneration
    
    if ($Script:NeedsCertRegeneration) {
        if (-not (New-SelfSignedCertificates)) {
            Read-Host "Press Enter to exit"
            return
        }
        $null = Install-CACertificate
        Save-State
    } else {
        Write-Log "Using existing certificates" "SUCCESS"
    }
    
    # Templates
    if (-not (Copy-TemplateFiles)) {
        Read-Host "Press Enter to exit"
        return
    }
    
    # Docker
    Stop-ConflictingContainers
    
    if (-not (Start-DockerServices)) {
        Read-Host "Press Enter to exit"
        return
    }
    
    Wait-ForServices
    Test-Connectivity
    Show-Status
    Show-AccessInfo
    Open-Browser
    
    Write-Log "Setup complete!" "SUCCESS"
    Write-Host ""
    Write-Host "Commands:" -ForegroundColor Yellow
    Write-Host "  docker compose logs -f       # View all logs" -ForegroundColor Gray
    Write-Host "  docker logs caddy -f         # View Caddy logs" -ForegroundColor Gray
    Write-Host "  docker compose down          # Stop all services" -ForegroundColor Gray
    Write-Host "  docker compose restart       # Restart services" -ForegroundColor Gray
    Write-Host ""
    Write-Host "Press Enter to exit..." -ForegroundColor Cyan
    Read-Host
}

Main